###############################################################################
#  Team Generator
#  by Josh Dahlby
###############################################################################

import Bladex
import EnemyTypes
import Actions
import ItemTypes
import math
import Sounds
import darfuncs
import ObjStore
import whrandom
import CharStats
import Reference
import dust
import copy

###############################################################################
# Constants
###############################################################################

# Percentage to use for getting the maximum amount of experience a spawned 
# enemy can use.
# (Player.ExperienceCost(level) * MAXEXP > Enemy.ExperienceReward(level))
MAXEXP = 0.3

# Percentage to use for getting the minimum amount of experience a spawned 
# enemy can use.  This is used to set the level of the enemy.
# (Player.ExperienceCost(level) * MINEXP < Enemy.ExperienceReward(level))
MINEXP = 0.1

# Percentage to use for getting a weapon.  This relates to how much of the
# players total hit points a weapon can do in a single hit.
# (Player.MAXLIFE * MAXDAMAGE < Weapon.Damage)
MAXDAMAGE = 0.15

# Percentage to use for getting the highest resistance shield that can be 
# used.  
# (MAXLIFE * MAXRES > Shield.Resistance)
MAXRES = 2

# Maximum level to spawn
MAXLEVEL = 19

###############################################################################
# Fix some entries for enemies that just don't seem right
###############################################################################

CharStats.CharExperienceReward['Salamander'] = [100, 200, 400, 800, 1600, 3200, 6400, 12800, 25600, 51200, 70000, 90000, 110000, 130000, 150000, 170000, 190000, 210000, 240000, 260000]
CharStats.CharMaxLifeValue['Ragnar'] = [60, 100, 200, 400, 600, 800, 1000, 1200, 1500, 1700, 1900, 2100, 2300, 2500, 2700, 2900, 3100, 3300, 3500, 3900]
CharStats.CharDamageData['Ragnar'] = [12, 15, 18, 25, 30, 40, 50, 60, 70, 80, 90, 100, 110, 120, 130, 140, 150, 160, 170, 180]
CharStats.CharExperienceReward['Ragnar'] = [40, 50, 60, 120, 240, 480, 960, 1920, 3840, 7680, 15360, 30720, 61440, 122880, 135000, 150000, 170000, 190000, 210000, 240000]

def GenBotExp():
	cost = CharStats.CharExperienceCost['Knight_N']
	exp = []
	for i in range(len(cost)):
		exp.append((cost[i] * MAXEXP) - 1)
	CharStats.CharExperienceReward['Knight_N'] = exp
	CharStats.CharExperienceReward['Barbarian_N'] = exp
	CharStats.CharExperienceReward['Amazon_N'] = exp
	CharStats.CharExperienceReward['Dwarf_N'] = exp

GenBotExp()

###############################################################################
# Weapon Types - Organized by type.  Each enemy is assigned a type
###############################################################################
W_NONE	= -1	# No weapons
W_1H	= 0     # One handed weapons
W_2H	= 1		# Two handed weapons
W_SP	= 3		# Spears
W_DAL	= 4		# DalGurak weapon
W_VAMP	= 5		# Vampire weapon
W_CHAOS = 6		# Chaos Knight weapon
W_BIG	= 7		# Troll and Minotaur weapons
W_OTHER	= 8		# Weapons not used
W_MAGIC = 9		# Magic 1 handed weapons

WeaponTypes = {}
WeaponTypes[W_NONE] = []
WeaponTypes[W_1H] = ["Gladius", "Sablazo", "Hacha", "Hacha2", "Hacha3", 
	"Hacha4", "Hacha5", "Hacha6", "Garrote", "Martillo", "Martillo2", 
	"Garropin", "MazaDoble", "Garrote2", "Martillo3", "Orksword", 
	"Espadaelfica", "Espadaromana", "Espadacurva", "Dagesse", "Cimitarra", 
	"EgyptSword", "Espadafilo", "Espada", "Maza", "Maza2", "Maza3", "TaiSword", 
	"LightEdge", "Ninjato", "HookSword", "Katana", "DoubleSword"]

WeaponTypes[W_MAGIC] = ["CrushHammer", "FireAxe", "IceHammer", "QueenSword", 
	"FireSword", "IceSword"]

WeaponTypes[W_2H] = ["Chaosword", "DeathSword", "LongSword", "Alfanje", 
	"BigSword", "SawSword", "FlatSword",  "FireBigSword", "IceAxe", "Eclipse", 
	"Guadanya", "Hacha2hojas", "RhinoClub", "Hacharrajada"]

WeaponTypes[W_BIG] = ["Hachacarnicero", "Mazapiedra"]
WeaponTypes[W_VAMP] = ["VampWeapon"]
WeaponTypes[W_DAL] = ["DalWeapon"] # "DalBlade" is created automatically 
WeaponTypes[W_CHAOS] = ["Espadon"]
WeaponTypes[W_SP] = ["IceWand", "SteelFeather", "FireBo", "Bo", "Lanza", 
	"Naginata", "Tridente", "Hachacuchilla", "Naginata2", "DeathBo", "CrushBo", 
	"LanzaAncha", "Axpear", "Arpon", "Bichero", "Crosspear"]

WeaponTypes[W_OTHER] = [
#These have no skins	
	"DeathKatar", "Chakram", "Katarmoon", "Chakram2", "Katar", "KatarDoble",
# These are just not useful
	"Varita7", "Varita6", "Varita5", "Varita2", "Varita1", "Daga", "Cuchillo", 
	"Phurbhu", "Baston3", "Suriken", "Dagarrojar", "Canica", "KingSword",
	"Alabarda", 
# Don't know
	"EspadaMagica1", "EspadaMagica2", "EspadaMagica3",
# Special DalBlade
	"DalBlade",
# Trap weapons
	"CuchillaFernando", "Pendulo", "PinchoManuel", "PinchoMiguel", 
	"Roca1Aurelio", "Pivote"
]

KnightWeapons = [("Gladius", ), ("Gladius", ), ("Gladius", ), ("Gladius", ), 
	("Maza", ), ("Maza", ), ("Espadaromana", ), ("Espadaromana", ), 
	("Espadaelfica", ), ("Maza2", ), ("QueenSword", "HookSword", ),	("Espadacurva", ), 
	("FireSword", ), ("Dagesse", ), ("Cimitarra", ), ("Maza3", ), ("DoubleSword", ),
	("IceSword", "Espadafilo", ), ("Espada", )]

BarbarianWeapons = [("Chaosword", ), ("Chaosword", ), ("Chaosword", ), ("Chaosword", ), 
	("Eclipse", ), ("Eclipse", ), ("DeathSword", ), ("DeathSword", ), ("Guadanya", ), 
	("LongSword", ), ("QueenSword", "Alfanje", "Alfanje", ), ("Hacha2hojas", ), 
	("IceAxe", "FireBigSword", ), ("FlatSword", ), ("BigSword", ), ("RhinoClub", ), 
	("RhinoClub", ), ("FireBigSword", "Hacharrajada", ), ("SawSword", )]

DwarfWeapons = [("Garrote",), ("Garrote",), ("Hacha",), ("Hacha",), ("Hacha5",),
	("Hacha5",), ("Garropin",), ("Garropin",), ("Hacha4",), ("Hacha3",), 
	("QueenSword", "Martillo",), ("Martillo2",), ("IceHammer",), ("Garrote2",), 
	("MazaDoble",), ("Hacha6",), ("CrushHammer",), ("Hacha2", "FireAxe",), ("Martillo3",)]

AmazonWeapons = [ ("Bo",), ("Bo",), ("Bo",), ("Bo",), ("Bichero",), ("Bichero",), ("Lanza",), 
	("Lanza",), ("Naginata",), ("Tridente",), ("QueenSword", "Axpear",), ("DeathBo",), 
	("IceWand",), ("Crosspear",), ("FireBo", "Hachacuchilla",), ("CrushBo",), ("Arpon",), 
	("SteelFeather", "Naginata2",), ("LanzaAncha", )]

###############################################################################
# Shield Types - Organized by type.  Each enemy is assigned a type
###############################################################################
S_NONE	= -1	# No shield
S_NORMAL = 0	# Normal shield selection
S_CHAOS = 1		# Chaos Knight shield
S_DAL	= 2		# DalGurak shield
S_VAMP	= 3		# Vampire shield
S_MAGIC = 4		# Magic shield

ShieldTypes = {}
ShieldTypes[S_NONE] = []
ShieldTypes[S_NORMAL] = ["Escudo1", "Escudo2", "Escudo3", "Escudo4", "Escudo5", 
	"Escudo6", "Escudo7", "Escudo8", "Escudo9", "KingShield"]
ShieldTypes[S_VAMP] = ["VampShield"]
ShieldTypes[S_DAL] = ["DalShield"]
ShieldTypes[S_CHAOS] = ["Escudon"]
ShieldTypes[S_MAGIC] = ["MagicShield"]

###############################################################################
# Armor Types - Bots get armor at a certain level
###############################################################################

ArmorTypes = {}
ArmorTypes["Amazon"] = {5: "ArmaduraAmazonaLigera" }
ArmorTypes["Knight"] = {5: "ArmaduraCaballeroLigera",
					    10: "ArmaduraCaballeroMedia",
						15: "ArmaduraCaballeroCompleta"}
ArmorTypes["Barbarian"] = {5: "ArmaduraBarbaroLigera" }
ArmorTypes["Dwarf"] = {5: "ArmaduraEnanoLigera",
					   10: "ArmaduraEnanoMedia"}

###############################################################################
# Enemy information
#
# Each enemy is listed with the following information
#   name	= The enemy name.  Used to create it.
#   mmp		= The bitmap file for the enemy
#   wflag	= Name of the weapons array to use for the enemy.  
#   sflag	= Name of the shield array to use for the enemy.  
#	points	= How many points one of these takes from the MAXPOINTS.
#	dusting = Should the enemy be dusted?  1 for yes (default), 0 for no
#
#   Internally it calculates the following
#   MinPlayerLevel = Minimum level it should show up
#   MaxPlayerLevel = Maximum level it should show up
###############################################################################

EnemyData = {}

class EnemyDataItem:
	initialized = 0
	def __init__(self, name, mmp, wflag=W_NONE, sflag=S_NONE, points=100, dusting=1):
		self.Name = name
		self.MMPName = "../../3dChars/" + mmp + ".mmp"
		self.WeaponFlag = wflag
		self.ShieldFlag = sflag
		self.Points = points
		self.AllowDusting = dusting
		self.CanBeInvisible = wflag != W_NONE
		if self.Name in ('Lich', 'Knight_Zombie'):
			self.CanBeInvisible = 0

		global EnemyData
		if EnemyData.has_key(self.Name):
			Reference.debugprint("ERROR: Enemy already in EnemyData array")
		else:
			EnemyData[self.Name] = self

		# All characters have the same requirements.
		cost = CharStats.CharExperienceCost["Knight_N"]
		
		# Determine the minimum and maximum level the enemy can appear on
		self.MinPlayerLevel = 0
		self.MaxPlayerLevel = 19

	def Initialize(self):
		if self.initialized == 1:
			return
		self.initialized = 1
		self.Dummy = Bladex.CreateEntity(self.Name + "_dummy", self.GetCoreName(), 100000, 100000, 100000, "Person")
		self.AfterCreate(self.Dummy)
		darfuncs.HideBadGuy(self.Name + "_dummy")

	def GetCoreName(self):
		return self.Name

	def AfterCreate(self, entity):
		EnemyTypes.EnemyDefaultFuncs(entity)

	def GetWeaponType(self, level):
		return None

	def GetShieldType(self, level, weapon):
		return None

class IceGolemDataItem (EnemyDataItem):
	def GetCoreName(self):
		return "Golem_stone"

	def AfterCreate(self, entity):
		EnemyDataItem.AfterCreate(self, entity)
		entity.Alpha=0.6
		entity.SelfIlum=0.8
		entity.MeshName="Golem_ice"
		entity.Data.Resistances= copy.copy(CharStats.GetCharResistances("Golem_ice"))
		entity.Data.StoneType="Piedra_Glm_ic"
		entity.Data.StoneAlpha=0.6
		entity.Data.StoneSelfIlum=0.8

class DalGurakDataItem (EnemyDataItem):
	def AfterCreate(self, entity):
		EnemyDataItem.AfterCreate(self, entity)
		entity.Data.Phase = 2

class DarkLordDataItem (EnemyDataItem):
	def AfterCreate(self, entity):
		EnemyDataItem.AfterCreate(self, entity)
		entity.ImDeadFunc = entity.Data.StdImDead

class GladiatorDataItem (EnemyDataItem):
	def __init__(self, name, mmp, wflag=W_NONE, sflag=S_NONE, points=100, dusting=1):
		EnemyDataItem.__init__(self, name, mmp, wflag, sflag, points, dusting)
		self.CanBeInvisible = 0

	def Initialize(self):
		if self.initialized == 0:
			if self.Name == "Barbarian_N":
				Bladex.ReadBitMap("../../Data/Icons/icono barbaro.bmp","BarbarianIcon")
				Reference.EnemiesDefaultScorerData['Barbarian_N']=("BarbarianIcon","Barbarian")
			elif self.Name == "Dwarf_N":
				Bladex.ReadBitMap("../../Data/Icons/icono enano.bmp","DwarfIcon")
				Reference.EnemiesDefaultScorerData['Dwarf_N']=("DwarfIcon","Dwarf")
			elif self.Name == "Amazon_N":
				Bladex.ReadBitMap("../../Data/Icons/icono amazona.bmp","AmazonIcon")
				Reference.EnemiesDefaultScorerData['Amazon_N']=("AmazonIcon","Amazon")
		EnemyDataItem.Initialize(self)

	def AfterCreate(self, entity):
		import GladTypes
		GladTypes.GladiatorDefaultFuncs(entity)

		# Check to see if the character should have armor
		enType = entity.Kind[:len(entity.Kind)-2]
		if enType in ("Barbarian", "Amazon", "Knight", "Dwarf"):
			armor = ArmorTypes[enType]
			# Subtract two levels so player can hurt them.
			for i in range(entity.Level - 2, -1, -1):
				if armor.has_key(i+1):
					self.CreateArmor(entity, armor[i+1])
					break

	def CreateArmor(self, entity, type):
		object_data = Reference.DefaultObjectData[type]
		entity.Data.armour_level=object_data[2]
		entity.Data.armour_prot_factor=object_data[3]
		if entity.MeshName[:len(entity.MeshName)-2] == entity.Kind[:len(entity.Kind)-2]:
			ct = Bladex.GetCharType(entity.CharType, entity.CharTypeExt)
			if object_data[2]==0:
				entity.SetMesh(ct.NoArmour)
			elif object_data[2]==1:
				entity.SetMesh(ct.LowArmour)
			elif object_data[2]==2:
				entity.SetMesh(ct.MedArmour)
			elif object_data[2]==3:
				entity.SetMesh(ct.HighArmour)

	def GetWeaponType(self, level):
		if self.Name == "Knight_N":
			return self.GetLevelWeapon(KnightWeapons, level)
		elif self.Name == "Barbarian_N":
			return self.GetLevelWeapon(BarbarianWeapons, level)
		elif self.Name == "Dwarf_N":
			return self.GetLevelWeapon(DwarfWeapons, level)
		elif self.Name == "Amazon_N":
			return self.GetLevelWeapon(AmazonWeapons, level)
		return None

	def GetShieldType(self, level, weapon):
		if self.Name in ("Knight_N", "Dwarf_N"):
			if level < 3:
				return "Escudo1" #300
			elif level < 6:
				return "KingShield" #1000
			elif level < 8:
				return "Escudo9" #2000
			elif level < 10:
				return "Escudo4" #2500
			elif level < 13:
				return "Escudo8" #3000
			elif level < 15:
				return "Escudo3" #4000
			elif level < 17:
				return "Escudo7" #5000
			else:
				return "Escudo6" #8000
		elif weapon == "QueenSword":
			return "Escudo8"
		return None

	def GetLevelWeapon(self, weapons, level):
		if level >= len(weapons):
			level = len(weapons) - 1
		sel = weapons[level]
		if len(sel) > 1:
			i = whrandom.randint(0, len(sel) - 1)	
			return sel[i]
		else:
			return sel[0]

# Type that gets created when an error occurs
ERROR_ENEMY_TYPE = "Cos"

# Create enemy information
EnemyDataItem("Spidersmall", "spd", W_NONE, S_NONE, 1)
EnemyDataItem("Cos", "cosita", W_NONE, S_NONE, 1)
EnemyDataItem("Ork", "ork", W_1H, S_NORMAL, 2)
#EnemyDataItem("Dark_Ork", "dork", W_1H, S_NORMAL, 2)
EnemyDataItem("Knight_Traitor", "tkn", W_1H, S_NORMAL, 2)
EnemyDataItem("Great_Ork", "gok", W_1H, S_NORMAL, 3)
EnemyDataItem("Dark_Knight", "DarkKnight", W_1H, S_NORMAL, 3)
EnemyDataItem("Lich", "lch", W_1H, S_NORMAL, 2, 0)
EnemyDataItem("Knight_Zombie", "zkn", W_1H, S_NORMAL, 2, 0)
EnemyDataItem("Skeleton", "skl", W_1H, S_NORMAL, 3, 0)
EnemyDataItem("Salamander", "slm", W_NONE, S_NONE, 3)
EnemyDataItem("Troll_Dark", "trl_dk", W_BIG, S_NONE, 4)
EnemyDataItem("Troll_snow", "trl_sn", W_BIG, S_NONE, 4)
EnemyDataItem("Minotaur", "min", W_BIG, S_NONE, 4)
EnemyDataItem("Golem_clay", "glm_cl", W_NONE, S_NONE, 4)
EnemyDataItem("Golem_stone", "glm_st", W_NONE, S_NONE, 4)
EnemyDataItem("Golem_metal", "glm_mt", W_NONE, S_NONE, 4)
EnemyDataItem("Little_Demon", "ldm", W_NONE, S_NONE, 4, 0)
EnemyDataItem("Ragnar", "rgn", W_1H, S_NORMAL, 2)
EnemyDataItem("Golem_lava", "glm_lv", W_NONE, S_NONE, 4)
DalGurakDataItem("DalGurak", "DalGurak", W_DAL, S_DAL, 6)
EnemyDataItem("Vamp", "vmp", W_VAMP, S_VAMP, 4)
EnemyDataItem("ChaosKnight", "chk", W_CHAOS, S_CHAOS, 4)
#EnemyDataItem("Great_Demon", "gdemon", W_NONE, S_NONE, 6, 0)
# Needs some work.  He disappears into nowhere.
#DarkLordDataItem("DarkLord", "darklord", W_NONE, S_NONE, 100)
# Special classes
IceGolemDataItem("Golem_ice", "glm_ic", W_NONE, S_NONE, 4)
# Other chars
#GladiatorDataItem("Enano1", "enanos", W_1H, S_NORMAL, 4)
#GladiatorDataItem("Enano2", "enanos", W_1H, S_NORMAL, 4)
#GladiatorDataItem("Gold_Ork", "org", W_1H, S_NORMAL, 4)
# Bots 
GladiatorDataItem("Knight_N", "KgtSkin2", W_1H, S_NORMAL, 6)
GladiatorDataItem("Barbarian_N", "BarSkin2", W_2H, S_NONE, 6)
GladiatorDataItem("Amazon_N", "AmzSkin1", W_SP, S_NONE, 6)
GladiatorDataItem("Dwarf_N", "DwfSkin1", W_1H, S_NORMAL, 6)

###############################################################################
# EnemyInfo
#
#   This class is used to hold a generated enemy's information.  It is used
#   internally by the EnemyGenerator
#
###############################################################################

enemyID = 0
class EnemyInfo:
	def __init__(self):
		global enemyID
		self.Name = None
		while self.Name == None:
			id = enemyID
			enemyID = enemyID + 1
			self.Name = "enemy_" + str(id)
			if Bladex.GetEntity(self.Name) != None:
				self.Name = None

		self.Data = None
		self.WeaponType = None
		self.WeaponName = None
		self.ShieldType = None
		self.ShieldName = None
		self.Position = [0, 0, 0] 
		self.Level = 0

###############################################################################
# EnemyGenerator
#
#   The EnemyGenerator class is used to randomly create multiple monsters 
#   based on the users level.
#
#	Creation Parameters:
#		playerName = Name of the player.  Usually "Player1"
#		getPosFunc = A method that is called to get a valid position for the
#			enemy.  Return a Position tuple (x,y,z)
#		levelUpFunc = A method that is called when the player levels up.
#			Return 0 to stop the generator, or 1 to continue
#		createEnemyFunc = A method that is called when an enemy is created.
#			This is called before the enemy is shown.  It takes 1 parameter
#			which is the enemy entity object.
#		showEnemyFunc =  A method that is called when an enemy is about to
#			be activated.  It takes 1 parameter which is the enemy entity 
#			object.
#
###############################################################################


class EnemyGenerator:

	def __init__(self, playerName="Player1", eList=[], getPosFunc=None, levelUpFunc=None, createEnemyFunc=None, showEnemyFunc=None, allowInvisible=1, mxEnemies=4):
		self.ObjId = ObjStore.GetNewId()
		self.Enemies = {}
		ObjStore.ObjectsStore[self.ObjId] = self
		self.player = Bladex.GetEntity(playerName)
		self.OnGetPosition = getPosFunc
		self.OnLevelUp = levelUpFunc
		self.OnCreateEnemy = createEnemyFunc
		self.OnShowEnemy = showEnemyFunc
		self.AllowInvisible = allowInvisible
		self.MaxEnemies = mxEnemies
		if self.MaxEnemies < 1:
			self.MaxEnemies = 1
		self.DoGeneration = 0
		self.Queue = []
		self.EnemyList = []

		global EnemyData
		if len(eList) == 0:
			eList = EnemyData.keys()
		self.EnemyList = eList
		# Make sure they are all valid
		for key in self.EnemyList:
			if not EnemyData.has_key(key):
				self.EnemyList.remove(key)
		for key in self.EnemyList:
			EnemyData[key].Initialize()
		self.ValidEnemies = []
		for key in self.EnemyList:
			self.ValidEnemies.append(EnemyData[key])

	def StartGeneration(self):
		self.DoGeneration = 1
		self.GenerateEnemies()

	def StopGeneration(self):
		self.DoGeneration = 0

	# This function manages the actual generation of the enemies.
	# It determines if an enemy should appear, and also how many should
	# appear
	def GenerateEnemies(self):
		if self.DoGeneration == 0:
			return

		global MonsterData
		curPoints = 0
		curNumber = 0
		list = self.Enemies.keys()
		for key in list:
			info = self.Enemies[key]
			curNumber = curNumber + 1
		
		for i in range(self.MaxEnemies):
			if curNumber == i:
				if len(self.Queue) == 0:
					for i in range(10): # generate several monsters at a time.
						self.GenerateSingleEnemy()
				self.ShowEnemy()
				curNumber = curNumber + 1

	def GetMinimumPlayerLevel(self):
		global EnemeyData
		minLevel = 19
		for enemy in self.EnemyList:
			lvl = EnemyData[enemy].MinPlayerLevel
			if lvl < minLevel:
				minLevel = lvl
		return minLevel

	# Does the work of actually generating the enemy
	def GenerateSingleEnemy(self):
		
		info = EnemyInfo()
		try:
			info.Data = self.GenEnemyType()
			info.Name = info.Data.Name + info.Name 
			info.Level = self.GenEnemyLevel(info)
			# See if the enemy specifies it's own weapon generation
			info.WeaponType = info.Data.GetWeaponType(info.Level)
			if info.WeaponType == None:
				info.WeaponType = self.GenEnemyWeapon(info)
			if info.WeaponType != None:
				info.WeaponName = info.Name + "_weapon"
			# See if the enemy specifies it's own shield generation
			info.ShieldType = info.Data.GetShieldType(info.Level, info.WeaponType)
			if info.ShieldType == None:
				info.ShieldType = self.GenEnemyShield(info)
			if info.ShieldType != None:
				info.ShieldName = info.Name + "_shield"
			info.Position = (0, 0, 0)
		except:
			import traceback
			traceback.print_exc()
			Reference.debugprint("ERROR: Exception while generating new enemy")
			info.Data = EnemyData[ERROR_ENEMY_TYPE]
			info.WeaponType = None
			info.ShieldType = None
			info.Position = (0, 0, 0)
			info.Level = 0

		self.CreateEnemy(info)

	# Trap the enemy death so we can generate a new enemy
	def EnemyDied(self, entity):
		enemy = Bladex.GetEntity(entity)
		if enemy == None:
			return
		if self.Enemies.has_key(entity):
			info = self.Enemies[entity]
			del self.Enemies[entity]
			global EnemyData
			if EnemyData[enemy.Kind].AllowDusting > 0:
				Bladex.AddScheduledFunc(Bladex.GetTime()+10, self.DustPerson,(entity,), "Dust_" + entity)
			del info
		Bladex.AddScheduledFunc(Bladex.GetTime()+1, self.GenerateEnemies,(), "GenerateEnemies")
		enemy.Data.OldImDeadFunc(entity)

	def DustPerson(self, entity):
		# Check to see it exists, before calling EnPolvoPerson
		body = Bladex.GetEntity(entity)
		if body != None:
			dust.EnPolvoPerson(entity, 100, 0)
			Bladex.AddScheduledFunc(Bladex.GetTime()+4, self.RemoveBody,(entity, ), "RemoveBody")
		

	def RemoveBody(self, entity):
		body = Bladex.GetEntity(entity)
		if body != None:
			body.SubscribeToList("Pin")

	# Create the actual enemy entity and its inventory	and put it in the queue
	def CreateEnemy(self, info):
		# This initial position isn't really needed, but it gets rid of the warning.
		pos = self.GenEnemyPosition()
		info.Enemy = Bladex.CreateEntity(info.Name, info.Data.GetCoreName(), pos[0], pos[1], pos[2], "Person")
		info.Enemy.Level = info.Level
		info.Data.AfterCreate(info.Enemy)
		info.Enemy.Life = CharStats.GetCharMaxLife(info.Enemy.Kind, info.Enemy.Level)
		info.Enemy.Data.OldImDeadFunc = info.Enemy.ImDeadFunc
		info.Enemy.ImDeadFunc = self.EnemyDied

		# Create shield
		if info.ShieldType != None:
			shield = ItemTypes.MakeShield(info.ShieldName, info.ShieldType)
			ItemTypes.ItemDefaultFuncs(shield)
			Actions.TakeObject(info.Enemy.Name, shield.Name)

		# Create Weapon
		if info.WeaponType != None:
			weapon = None
			if info.WeaponType == "VampWeapon":
				weapon = ItemTypes.MakeVampireSword(info.WeaponName)
			else:
				weapon = Bladex.CreateEntity(info.WeaponName, info.WeaponType, 0, 0, 0, "Weapon")
			ItemTypes.ItemDefaultFuncs(weapon)
			Actions.TakeObject(info.Enemy.Name, weapon.Name)
		if self.OnCreateEnemy != None:
			apply(self.OnCreateEnemy, (info.Enemy, ))
		darfuncs.HideBadGuy(info.Enemy.Name)
		self.Queue.append(info)

	def SetEnemyPosition(self, info):
		info.Enemy.Position = self.GenRandomPosition(info)
		info.Enemy.InitPos = info.Enemy.Position
		Actions.TurnToFaceEntityNow(info.Enemy.Name, self.player.Name)
		darfuncs.UnhideBadGuy(info.Enemy.Name)
		info.Enemy.SetOnFloor()
		if not info.Enemy.CanISee(self.player):
			Reference.debugprint("CANNOT SEE ENEMY")

	def GenRandomPosition(self, info):
		for i in range(50):
			pos = self.GenEnemyPosition()
			if info.Enemy.TestPos(pos[0], pos[1], pos[2], 0, 10000):
				return pos
		Reference.debugprint("POS FAILED")
		return (0,0,0)

	# Show the enemy on the map
	def ShowEnemy(self):
		if self.DoGeneration != 1:
			return
		info = self.Queue.pop(0)
		if info == None:
			Reference.debugprint("ERROR: ShowEnemy called, but nothing to show")
			return
		if self.AllowInvisible and info.Data.CanBeInvisible:
			chance = self.player.Level * 0.7
			if chance > 10:
				chance = 10
			if whrandom.randint(0, 100) <= chance:
				info.Enemy.Alpha = 0
				info.Enemy.CastShadows = 0
		self.SetEnemyPosition(info)
		info.Enemy.Data.JoinGroup(info.Enemy.Name, "generated" + str(self.ObjId))
		info.Enemy.Data.LaunchMyWatch(info.Enemy.Name)
		self.Enemies[info.Enemy.Name] = info
		#info.Enemy.SetActiveEnemy(self.player.Name)
		if self.OnShowEnemy:
			self.OnShowEnemy(info.Enemy);
		
	# Generate a random enemy type
	def GenEnemyType(self):
		return self.ValidEnemies[whrandom.randint(0, len(self.ValidEnemies)-1)]

	# Generate a random weapon for the specified enemy
	def GenEnemyWeapon(self, info):
		global MAXDAMAGE
		weapons = WeaponTypes[info.Data.WeaponFlag]
		if len(weapons) == 0:
			return None
		if len(weapons) == 1:
			return weapons[0]
		maxdamage = CharStats.GetCharMaxLife(self.player.Kind, self.player.Level) * MAXDAMAGE
		mindamage = CharStats.GetCharDefenseData(self.player.CharType, self.player.Level) + self.player.Data.armour_prot_factor + 2

		valid = []
		for weapon in weapons:
			dmg = Reference.DefaultObjectData[weapon][1]
			if dmg <= maxdamage and dmg >= mindamage:
				valid.append(weapon)
		if len(valid) == 0:
			return weapons[0]
		if len(valid) == 1:
			return valid[0]
		return valid[whrandom.randint(0, len(valid) - 1)]

	# Generate a random shield for the specified enemy
	def GenEnemyShield(self, info):
		global MAXRES
		shields = ShieldTypes[info.Data.ShieldFlag]
		if len(shields) == 0:
			return None
		if len(shields) == 1:
			return shields[0]
		maxres = CharStats.GetCharMaxLife(self.player.Kind, self.player.Level) * MAXRES
		minres = CharStats.GetCharMaxLife(self.player.Kind, self.player.Level) * 0.4

		valid = []
		for shield in shields:
			res = Reference.DefaultObjectData[shield][2]
			if res <= maxres and res >= minres:
				valid.append(shield)
		if len(valid) == 0:
			return shields[0]
		if len(valid) == 1:
			return valid[0]
		return valid[whrandom.randint(0, len(valid) - 1)]

	# Get a valid position for the enemy
	def GenEnemyPosition(self):
		if self.OnGetPosition != None:
			return apply(self.OnGetPosition, ())
		else:
			return (0, 0, 0)

	# Generate the level of the enemy
	def GenEnemyLevel(self, info):
		global MAXEXP
		global MINEXP
		global MAXLEVEL

		exp = CharStats.GetCharExperienceCost(self.player.Kind, self.player.Level)
		maxexp = exp * MAXEXP
		minexp = exp * MINEXP
		minlevel = 0
		maxlevel = 0
		for x in CharStats.CharExperienceReward[info.Data.GetCoreName()]:
			if x < minexp:
				minlevel = minlevel + 1
			if x > maxexp:
				maxlevel = maxlevel - 1
				break
			maxlevel = maxlevel + 1
		else:
			maxlevel = MAXLEVEL
		
		if self.player.Level < 4 and info.Name == 'Lich':
			maxlevel = self.player.Level
		if minlevel < 0:
			minlevel = 0
		if maxlevel < 0:
			maxlevel = 0
		if minlevel > maxlevel:
			minlevel = maxlevel
		return whrandom.randint(minlevel, maxlevel)
		
	def CheckEnemies(self):
		list = self.Enemies.keys()
		for key in list:
			info = self.Enemies[key]
			if not info.Enemy.CanISee(self.player):
				info.Enemy.Life = 0
